% Copyright (C) 2023 Computer Vision Lab, Electrical Engineering, 
% Indian Institute of Science, Bengaluru, India.
% All rights reserved.
% 
% Redistribution and use in source and binary forms, with or without
% modification, are permitted provided that the following conditions are
% met:
%     * Redistributions of source code must retain the above copyright
%       notice, this list of conditions and the following disclaimer.
%     * Redistributions in binary form must reproduce the above
%       copyright notice, this list of conditions and the following
%       disclaimer in the documentation and/or other materials provided
%       with the distribution.
%     * Neither the name of Indian Institute of Science nor the
%       names of its contributors may be used to endorse or promote products
%       derived from this software without specific prior written permission.
% 
% THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS 
% OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
% FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL 
% THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER 
% LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING 
% FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER 
% DEALINGS IN THE SOFTWARE.
% 
% Author: Lalit Manam
% 
% This file is a part of the implementation for the paper:
% Lalit Manam and Venu Madhav Govindu, Sensitivity in Translation Averaging,
% Neural Information Processing Systems, 2023.

% Demo for filtering skewed triangles
clear; close all;

%% Load data
% The data should contain the following:
% RT: 3 x M matrix containing the relative directions all of which are
% placed in the same coordinate frame.
% edges: M X 2 matrix containing the edges list.

load('demo_data.mat');

%% Filter parameters
filterTripletsTh=5;

%% Remove the skewed triangles
RT=RT./vecnorm(RT,2,1);

% Get the triplets
G=graph(edges(:,1),edges(:,2),1:size(edges,1));
[~,edgecycles]=allcycles(G,'MaxCycleLength',3);

% Getting the boundary matrix
numTriplets=size(edgecycles,1);
B2_ridx=G.Edges.Weight(cell2mat(edgecycles));
B2_cidx=repmat([1:numTriplets]',1,3); 
B2_val=ones(numTriplets,3);
B2=sparse(B2_ridx,B2_cidx,B2_val);
maskMatrix=B2*B2';

% Get the dot products for angle matrix
maskMatrix(maskMatrix>0)=1;
maskMatrix=maskMatrix-diag(diag(maskMatrix));
maskMatrix=triu(maskMatrix);
[spM_ridx,spM_cidx]=find(maskMatrix);
spM_val=sum(RT(:,spM_ridx).*RT(:,spM_cidx),1);

% Correct the signs for the dot products
e1=edges(spM_ridx,:); e2=edges(spM_cidx,:);
signChangeEle=e1(:,1)==e2(:,2)|e1(:,2)==e2(:,1);
spM_val(signChangeEle)=-spM_val(signChangeEle);
spM_val=acos(spM_val);

% Get the angle matrix
numEdges=size(edges,1);
angleMatrix=sparse(spM_ridx,spM_cidx,spM_val,numEdges,numEdges);
angleMatrix=angleMatrix+angleMatrix';

clear spM_ridx spM_cidx spM_val B2_ridx B2_cidx B2_val B2 maskMatrix e1 e2 signChangeEle

% Get the condition number before filtering
condNoBeforeFilter=condest(angleMatrix);
disp(['Condition number of angle matrix before filtering: ',num2str(condNoBeforeFilter,'%1.1e')]);

% Get all angles
edgecycles=cell2mat(edgecycles);
tempedges=G.Edges.Weight(edgecycles);
linearidx12=sub2ind(size(angleMatrix),tempedges(:,1),tempedges(:,2));
linearidx23=sub2ind(size(angleMatrix),tempedges(:,2),tempedges(:,3));
linearidx13=sub2ind(size(angleMatrix),tempedges(:,1),tempedges(:,3));

allAngles=[full(angleMatrix(linearidx12)),full(angleMatrix(linearidx23)),full(angleMatrix(linearidx13))];
allAngles=allAngles*180/pi;
clear linearidx12 linearidx23 linearidx13

% Filter bad triplets
disp('Filtering Triplets based on angles');

% Remove the bad triplets
tripletIdxRem=any(allAngles<filterTripletsTh,2);
edgecycles(tripletIdxRem,:)=[];
ret_edges=false(size(edges,1),1);
ret_edges(unique(edgecycles(:)))=true;
eidx=~ret_edges;

edges(eidx,:)=[];
RT(:,eidx)=[];

% Extract the largest connected Parallel Rigid component
G=graph(edges(:,1),edges(:,2),1:size(edges,1));
[cycles,edgecycles]=allcycles(G,'MaxCycleLength',3);

cycles=cell2mat(cycles);
edgecycles=cell2mat(edgecycles);
numEdges=size(edges,1);
tripletGraphEdges=cell(numEdges,1);

p=gcp('nocreate');
if(isempty(p))
    p=gcp;
end

parfor i=1:numEdges
    connNodes=find(i==edgecycles(:,1)|i==edgecycles(:,2)|i==edgecycles(:,3));
    if(numel(connNodes)>1)
        tripletGraphEdges{i}=[connNodes(1:end-1),connNodes(2:end)];
    end
end
tripletGraphEdges=cell2mat(tripletGraphEdges);
tripletGraphEdges=unique(tripletGraphEdges,'rows');

% Extract largest connected component of the Triplet graph
G_T=graph(tripletGraphEdges(:,1),tripletGraphEdges(:,2));
bins=conncomp(G_T,'OutputForm','vector');
nodes=find(bins==mode(bins)); 
edgecycles_PR=edgecycles(nodes,:);
edges_idx_PR=unique(edgecycles_PR(:));
eidx=setdiff([1:numEdges],edges_idx_PR);

edges(eidx,:)=[];
RT(:,eidx)=[];
clear G_T edgecycles_PR edges_idx_PR

% Get the conditioning of the system using the angle matrix
% Get the triplets
G=graph(edges(:,1),edges(:,2),1:size(edges,1));
[~,edgecycles]=allcycles(G,'MaxCycleLength',3);

% Get the angles and create a sparse matrix of angles in triangles
RT=RT./vecnorm(RT,2,1);
numTriplets=size(edgecycles,1);
B2_ridx=G.Edges.Weight(cell2mat(edgecycles));
B2_cidx=repmat([1:numTriplets]',1,3); 
B2_val=ones(numTriplets,3);
B2=sparse(B2_ridx,B2_cidx,B2_val);
maskMatrix=B2*B2';

% Get the dot products for angle matrix
maskMatrix(maskMatrix>0)=1;
maskMatrix=maskMatrix-diag(diag(maskMatrix));
maskMatrix=triu(maskMatrix);
[spM_ridx,spM_cidx]=find(maskMatrix);
spM_val=sum(RT(:,spM_ridx).*RT(:,spM_cidx),1);

% Correct the signs for the dot products
e1=edges(spM_ridx,:); e2=edges(spM_cidx,:);
signChangeEle=e1(:,1)==e2(:,2)|e1(:,2)==e2(:,1);
spM_val(signChangeEle)=-spM_val(signChangeEle);
spM_val=acos(spM_val);
% Get the angle matrix
numEdges=size(edges,1);
angleMatrix=sparse(spM_ridx,spM_cidx,spM_val,numEdges,numEdges);
angleMatrix=angleMatrix+angleMatrix';
clear spM_ridx spM_cidx spM_val B2_ridx B2_cidx B2_val B2 maskMatrix e1 e2 signChangeEle

% Get the conditioning of the system
condNoAfterFilter=condest(angleMatrix);
disp(['Condition number of angle matrix after filtering: ',num2str(condNoAfterFilter,'%1.1e')]);
