% Copyright (C) 2021 Computer Vision Lab, Indian Institute of Science, Bengaluru (IISc).
% All rights reserved.
%
% This implementation is based on the paper 
% "It's All In The Weights: Robust Rotation Averaging Revisited" by
% Chitturi Sidhartha, Venu Madhav Govindu.
%
% AUTHOR: CHITTURI SIDHARTHA
%
% Redistribution and use in source and binary forms, with or without
% modification, are permitted provided that the following conditions are
% met:
% 
%     * Redistributions of source code must retain the above copyright
%       notice, this list of conditions and the following disclaimer.
% 
%     * Redistributions in binary form must reproduce the above
%       copyright notice, this list of conditions and the following
%       disclaimer in the documentation and/or other materials provided
%       with the distribution.
% 
%     * Neither the name of the Indian Institute of Science, Bengaluru nor the
%       names of its contributors may be used to endorse or promote products
%       derived from this software without specific prior written permission.
% 
% THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
% AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
% IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
% ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDERS OR CONTRIBUTORS BE
% LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
% CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
% SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
% INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
% CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
% ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
% POSSIBILITY OF SUCH DAMAGE.
%
% This code is revised from the code of AVISHEK CHATTERJEE based on the
% paper "Robust Relative Rotation Averaging", TPAMI-2018
%
%
% function [R] = RobustMeanSO3Graph_GOM(RR,I,SIGMA,[Rinit],[maxIters],type)
% INPUT:    RR = 'm' number of 3 X 3 Relative Rotation Matrices (R_ij) 
%                    stacked as a 3 X 3 X m Matrix
%                    OR
%                    'm' number of 4 X 1 Relative Quaternions (R_ij) 
%                    stacked as a 4 X m  Matrix
%              I = Index matrix (ij) of size (2 X m) such that RR(:,:,p)
%                    (OR RR(:,p) for quaternion representation)  is
%                    the relative rotation from R(:,:,I(1,p)) to R(:,:,I(2,p))
%                    (OR R(:,I(1,p)) and  R(:,I(2,p)) for quaternion representation)
%                    It is recommended that edge indices be sorted in row-major order. 
%              n_feat = m x 1 matrix where n_feat(p) equals the number of
%                       corrrespondences (matched features) between the cameras I(1,p) and I(2,p).
%              Rinit = Optional initial guess. 
%                        Put [] to automatically comput Rinit from spanning tree
%              maxIters = Maximum number of iterations. Default 100
%              type   =  Loss function (Geman-McClure, Huber etc..)
%                           Default: Geman-McClure.
% OUTPUT: R  = 'n' number of 3 X 3 Absolute Rotation matrices stacked as
%                    a  3 X 3 X n Matrix 
%                    OR
%                    'n' number of 4 X 1 Relative Quaternions (R_ij) 
%                    stacked as a 4 X n  Matrix
%
% IMPORTANT NOTES:
% The underlying model or equation is assumed to be:
% X'=R*X; Rij=Rj*inv(Ri) i.e. camera centered coordinate system is used
% and NOT the geocentered coordinate for which the underlying equations are
% X'=inv(R)*X; Rij=inv(Ri)*Rj. 
% To use geocentered coordinate please transpose the rotations or change
% the sign of the scalar term of the quaternions before feeding into the
% code and also after getting the result.
%
% Feeding of not connected graph is not recomended.
%
% This code is able to handle inputs in both Rotation matrix as well as
% quaternion format. The Format of output is same as that of the input.


function  [R,Iteration]=RobustMeanSO3Graph_GOM(RR,I,n_feat, Rinit,maxIters,type)
n_feat_avail = true;
if(nargin<6 || isempty(type)); type='Geman-McClure';end
if(nargin<5 || isempty(maxIters));maxIters=100;end
if(nargin<3 || isempty(n_feat)); n_feat_avail = false; end

N=max(max(I));%Number of cameras or images or nodes in view graph

QuaternionIP=(size(RR,2)==4);

if(~QuaternionIP)
    QQ=[RR(1,1,:)+RR(2,2,:)+RR(3,3,:)-1, RR(3,2,:)-RR(2,3,:),RR(1,3,:)-RR(3,1,:),RR(2,1,:)-RR(1,2,:)]/2;
    QQ=reshape(QQ,4,size(QQ,3),1)';
    QQ(:,1)=sqrt((QQ(:,1)+1)/2);
    QQ(:,2:4)=(QQ(:,2:4)./repmat(QQ(:,1),[1,3]))/2;    
else
    QQ=RR;  
end

if(nargin>3  && (~isempty(Rinit)))
    if(size(Rinit,1)==3)
        Q = rotm2quat(Rinit);        
    else
        Q=Rinit;
    end
else
    if(n_feat_avail)
        n_feat = -1-n_feat;  %to get maximal spanning tree using minspantree

        G = graph(I(1,:),I(2,:),n_feat);
        key_converter = @(i,j)sub2ind([N,N],i,j);
        M = containers.Map(key_converter(I(1,:),I(2,:)),1:size(I,2));

        T = minspantree(G);
        T_edges = T.Edges.EndNodes;
        span_ids = cell2mat(values(M,num2cell(key_converter(T_edges(:,1),T_edges(:,2)))));
        QQ2 = QQ(span_ids,:);
        Q = compute_span_tree_init(QQ2,T_edges');  
    else
        Q = compute_span_tree_init(QQ, I);
    end
end

% Formation of A matrix.
m=size(I,2);
i=[[1:m];[1:m]];i=i(:);
j=I(:);
s=repmat([-1;1],[m,1]);
k=(j~=1);
Amatrix=sparse(i(k),j(k)-1,s(k),m,N-1);
W=zeros(N,4);

Iteration=0;

disp(num2str([0 NaN toc]));


tau = 3; tun = 16;
alpha = pi/360; % Minimum value of tun.
e2prev = zeros(m,1)+5;

eta = 0.5; 
ind_red = [];count=0;

i=I(1,:);j=I(2,:);
delE = quatmultiply(quatinv(Q(j,:)),quatmultiply(QQ,Q(i,:)));

while(Iteration<maxIters && tun>=alpha)
    B = quat2axang(delE);
    B = B(:,1:3).*B(:,4);
    
    B(isnan(B))=0;
    W(1,:)=[1 0 0 0];
    
    E=(0*Amatrix*W(2:end,2:4)-B); 
    e2 = sqrt(sum(E.^2,2));
    
     if(Iteration==0)
        knee = prctile(e2,95);
        tun = max(knee*sqrt(3),0.01);
     end
    
    ind_g = find(e2>e2prev);
    ind_r = setdiff(1:m,ind_g);

    ind_g = setdiff(ind_g,ind_red);
    ind_r = setdiff(ind_r,ind_red);
  
    [fprev,~] = compute_lossfn_values(type,e2prev,tun);
    [f,Weights] = compute_lossfn_values(type,e2,tun);
    
    leq_prev = sum(fprev(ind_r));
    leq_curr = sum(f(ind_r));
    great_prev = sum(fprev(ind_g));
    great_curr = sum(f(ind_g));
    
    
    del_leq_k = leq_prev-leq_curr;
    del_great_k = great_curr-great_prev;
    
    %stopping criterion
    rho = (del_leq_k - del_great_k)/(del_leq_k + del_great_k);
    e2prev = e2;

    % Filtering of outlier edges.
    tau1 = max(tun,0.08);
    ind_red = find(e2>tau1);
    Weights(e2>tau1) = Weights(e2>tau1)/100;

    
    if(rho<=eta)
        tun = tun/tau;
        count= count+1;
        nless = sum(e2 < tun/sqrt(3));
        nequal = sum(e2 == tun/sqrt(3));
        centile = 100 * (nless + 0.5*nequal) / length(e2);
        if(centile>=95)
            eta = 0.5;
        elseif(centile>=90 && centile<95)
            eta = 0.25;
        elseif(centile>=80 && centile<90)
            eta = 0.125;
        elseif(centile>=70 && centile<80)
            eta = 0.1;
        else
            eta = 0.1;
        end
        continue;
    end
    

    Anew = sparse(1:length(Weights),1:length(Weights),Weights,length(Weights),length(Weights))*Amatrix;
    Bnew = repmat(Weights,[1,size(B,2)]).*B;
    
    ATA = Anew'*Anew;
    W(2:end, 2:4) = (ATA)\(Anew'*Bnew);
    Wang = sqrt(sum(W(:,2:4).^2,2));
    Wax = W(:,2:4)./Wang;
    Wang = mod(Wang,pi);
    ind = find(Wang==0);
    Wax(ind,:) = repmat([1,0,0],length(ind),1);
    delQ = axang2quat([Wax,Wang]);
    qijdelv = quatmultiply(quatinv(delQ(j,:)),quatmultiply(delE,delQ(i,:)));

    Q = quatmultiply(Q,delQ);
    delE = qijdelv;
    Iteration=Iteration+1;    
end

disp(num2str([Iteration]));

if(~QuaternionIP)
    R=zeros(3,3,N);
    for i=1:size(Q,1)
        R(:,:,i)=quat2rotm(Q(i,:));
    end
else
    R=Q;
end

if(Iteration>=maxIters);disp('Max iterations reached');end

end
